<?php

namespace App\Http\Controllers;

use App\Models\CompanyProfile;
use App\Models\CompanyFeatureUsage;
use App\Models\Plan;
use App\Models\Subscription;
use App\Models\SubscriptionHistory;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Carbon;
use App\Models\Job;
use App\Models\JobApplication;
use App\Models\Interview;
use Illuminate\Support\Facades\File; // For direct filesystem operations
use Illuminate\Support\Facades\Log; // Added to fix Undefined type 'Log'

class CompanyProfileController extends Controller
{
    public function index(): JsonResponse
    {
        $profiles = CompanyProfile::with(['user'])->get();
        return response()->json([
            'status' => 'success',
            'data' => $profiles,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $profile = CompanyProfile::with(['user'])->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $profile,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'User_id' => 'required|exists:users,User_id',
            'Company_name' => [
                'required',
                'string',
                'max:255',
            ],
            'Description' => 'nullable|string',
            'family_details' => 'sometimes|array',
            'Logo_url' => 'nullable|file|mimes:jpg,jpeg,png|max:10240',
            'Location' => 'nullable|string|max:255',
            'first_name' => 'nullable|string|max:100',
            'last_name' => 'nullable|string|max:100',
            'postal_code' => 'nullable|string|max:10',
            'type_of_care_needed' => 'sometimes|array',
            'availability_needs' => 'sometimes|array',
            'special_needs' => 'nullable|string',
        ], [
            'Logo_url.file' => 'The logo must be a valid file.',
            'Logo_url.mimes' => 'The logo must be a JPG, JPEG, or PNG file.',
            'Logo_url.max' => 'The logo must not exceed 10MB.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $data = $request->only([
            'User_id',
            'Company_name',
            'Description',
            'family_details',
            'Location',
            'first_name',
            'last_name',
            'postal_code',
            'type_of_care_needed',
            'availability_needs',
            'special_needs',
        ]);

        if ($request->hasFile('Logo_url') && $request->file('Logo_url')->isValid()) {
            $logo = $request->file('Logo_url');
            $logoName = time() . '_logo_' . $logo->getClientOriginalName();
            $logoDir = public_path('storage/logos');

            if (!File::exists($logoDir)) {
                File::makeDirectory($logoDir, 0755, true);
                Log::info('Created directory: ' . $logoDir);
            }

            $logo->move($logoDir, $logoName);
            $logoPath = public_path('storage/logos/' . $logoName);

            if (!File::exists($logoPath)) {
                Log::error('Failed to store logo at: ' . $logoPath);
                return response()->json([
                    'status' => 'error',
                    'message' => 'Failed to store logo file',
                ], 500);
            }

            Log::info('Logo stored at: ' . $logoPath);
            $data['Logo_url'] = url('storage/logos/' . $logoName);
            Log::info('Generated Logo_url: ' . $data['Logo_url']);
        }

        DB::beginTransaction();
        try {
            $profile = CompanyProfile::create($data);

            DB::commit();
            return response()->json([
                'status' => 'success',
                'data' => $profile,
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to store company profile: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to store company profile: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function update(Request $request, $id): JsonResponse
    {
        try {
            $profile = CompanyProfile::findOrFail($id);

            $validator = Validator::make($request->all(), [
                'User_id' => 'sometimes|exists:users,User_id',
                'Company_name' => [
                    'sometimes',
                    'string',
                    'max:255',
                ],
                'Description' => 'sometimes|string',
                'family_details' => 'sometimes|array',
                'Logo_url' => 'sometimes|string',
                'Location' => 'sometimes|string|max:255',
                'first_name' => 'nullable|string|max:100',
                'last_name' => 'nullable|string|max:100',
                'postal_code' => 'nullable|string|max:10',
                'type_of_care_needed' => 'sometimes|array',
                'availability_needs' => 'sometimes|array',
                'special_needs' => 'nullable|string',
            ], [
                'Logo_url.string' => 'The logo must be a valid URL or base64 string.',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'errors' => $validator->errors(),
                ], 422);
            }

            $data = $request->only([
                'User_id',
                'Company_name',
                'Description',
                'family_details',
                'Location',
                'first_name',
                'last_name',
                'postal_code',
                'type_of_care_needed',
                'availability_needs',
                'special_needs',
            ]);

            if ($request->has('Logo_url') && !empty($request->input('Logo_url'))) {
                $logoInput = $request->input('Logo_url');
                if (preg_match('/^data:image\/(jpeg|jpg|png);base64,/', $logoInput)) {
                    $base64String = preg_replace('/^data:image\/(jpeg|jpg|png);base64,/', '', $logoInput);
                    if (!$this->isValidBase64($base64String)) {
                        return response()->json([
                            'status' => 'error',
                            'message' => 'Invalid base64 string for logo',
                        ], 422);
                    }
                    $logoData = base64_decode($base64String, true);
                    if ($logoData === false) {
                        return response()->json([
                            'status' => 'error',
                            'message' => 'Failed to decode base64 string for logo',
                        ], 422);
                    }
                    if (strlen($logoData) > 10240 * 1024) {
                        return response()->json([
                            'status' => 'error',
                            'message' => 'Logo file size exceeds 10MB',
                        ], 422);
                    }

                    if ($profile->Logo_url) {
                        $oldLogoPath = str_replace(url('storage/'), public_path('storage/'), $profile->Logo_url);
                        if (File::exists($oldLogoPath)) {
                            File::delete($oldLogoPath);
                            Log::info('Old logo deleted: ' . $oldLogoPath);
                        }
                    }

                    $logoExtension = [
                        'image/jpeg' => 'jpg',
                        'image/jpg' => 'jpg',
                        'image/png' => 'png'
                    ][$this->getMimeTypeFromBase64($logoInput)] ?? 'jpg';
                    $logoName = time() . '_logo.' . $logoExtension;
                    $logoDir = public_path('storage/logos');

                    if (!File::exists($logoDir)) {
                        File::makeDirectory($logoDir, 0755, true);
                        Log::info('Created directory: ' . $logoDir);
                    }

                    $logoPath = $logoDir . '/' . $logoName;
                    file_put_contents($logoPath, $logoData);

                    if (!File::exists($logoPath)) {
                        Log::error('Failed to store logo at: ' . $logoPath);
                        return response()->json([
                            'status' => 'error',
                            'message' => 'Failed to store logo file',
                        ], 500);
                    }

                    Log::info('Logo stored at: ' . $logoPath);
                    $data['Logo_url'] = url('storage/logos/' . $logoName);
                    Log::info('Generated Logo_url: ' . $data['Logo_url']);
                } elseif (filter_var($logoInput, FILTER_VALIDATE_URL) && strpos($logoInput, url('storage/logos/')) === 0) {
                    $data['Logo_url'] = $logoInput;
                } else {
                    Log::error('Invalid logo input: ' . $logoInput);
                    return response()->json([
                        'status' => 'error',
                        'message' => 'Logo must be a valid base64 string or storage URL',
                    ], 422);
                }
            }

            $profile->update($data);

            return response()->json([
                'status' => 'success',
                'data' => $profile,
            ], 200);
        } catch (\Exception $e) {
            Log::error('Failed to update company profile: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to update company profile: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function destroy($id): JsonResponse
    {
        $profile = CompanyProfile::findOrFail($id);
        $user = $profile->user;

        if ($profile->Logo_url) {
            $logoPath = str_replace(url('storage/'), public_path('storage/'), $profile->Logo_url);
            if (File::exists($logoPath)) {
                File::delete($logoPath);
                Log::info('Logo deleted: ' . $logoPath);
            }
        }

        $profile->delete();

        if ($user) {
            $user->delete();
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Company profile and associated user deleted successfully',
        ], 200);
    }

    public function deleteLogo($id): JsonResponse
    {
        $profile = CompanyProfile::findOrFail($id);

        if (!$profile->Logo_url) {
            return response()->json([
                'status' => 'error',
                'message' => 'No logo found in profile.',
            ], 404);
        }

        $logoPath = str_replace(url('storage/'), public_path('storage/'), $profile->Logo_url);
        if (File::exists($logoPath)) {
            File::delete($logoPath);
            Log::info('Logo deleted: ' . $logoPath);
        } else {
            Log::warning('Logo file not found: ' . $logoPath);
        }

        $profile->update(['Logo_url' => null]);

        return response()->json([
            'status' => 'success',
            'message' => 'Logo deleted successfully',
            'data' => $profile,
        ], 200);
    }

    private function getMimeTypeFromBase64($base64String)
    {
        $data = explode(';', $base64String)[0];
        return str_replace('data:', '', $data);
    }

    private function isValidBase64($string)
    {
        if (!preg_match('/^[A-Za-z0-9+\/=]+$/', $string)) {
            return false;
        }
        if (strlen($string) % 4 !== 0) {
            return false;
        }
        return true;
    }

    public function getStats($Company_id): JsonResponse
    {
        $profile = CompanyProfile::findOrFail($Company_id);

        $activeJobs = Job::where('Company_id', $Company_id)
            ->where('Status', 'active')
            ->count();

        $jobIds = Job::where('Company_id', $Company_id)
            ->pluck('Job_id')
            ->toArray();

        $totalApplications = JobApplication::whereIn('Job_id', $jobIds)
            ->count();

        $candidatesHired = JobApplication::whereIn('Job_id', $jobIds)
            ->where('Status', 'accepted')
            ->count();

        $applicationIds = JobApplication::whereIn('Job_id', $jobIds)
            ->pluck('Application_id')
            ->toArray();

        $interviews = Interview::whereIn('Application_id', $applicationIds)
            ->groupBy('Status')
            ->select('Status', DB::raw('count(*) as count'))
            ->pluck('count', 'Status')
            ->toArray();

        $interviewStats = [
            'scheduled' => $interviews['scheduled'] ?? 0,
            'completed' => $interviews['completed'] ?? 0,
            'cancelled' => $interviews['cancelled'] ?? 0,
            'mock' => $interviews['mock'] ?? 0,
        ];

        return response()->json([
            'status' => 'success',
            'data' => [
                'active_jobs' => $activeJobs,
                'total_applications' => $totalApplications,
                'interviews' => $interviewStats,
                'candidates_hired' => $candidatesHired,
            ],
        ], 200);
    }
}
